<!-- Put directly below your form (in the same blade) -->
<style>
    /* tiny helpers */
    .card-preview {
        max-width: 360px;
        margin: .75rem auto;
        padding: .9rem;
        border-radius: .6rem;
        background: linear-gradient(135deg, #0d6efd, #6610f2);
        color: #fff;
        box-shadow: 0 6px 18px rgba(13, 110, 253, 0.15);
        font-family: system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial;
    }

    .card-line {
        letter-spacing: .12em;
        font-weight: 600;
    }

    .field-error {
        font-size: .85rem;
        color: #dc3545;
        margin-top: .25rem;
    }

    .brand-badge {
        font-size: .9rem;
        font-weight: 700;
        padding: .15rem .5rem;
        border-radius: .25rem;
        background: rgba(255, 255, 255, 0.12);
    }
</style>

<!-- Optional visual preview (remove if you don't want one) -->
<div id="cardPreview" class="card-preview d-none">
    <div class="d-flex justify-content-between align-items-center mb-3">
        <div class="brand-badge" id="cardBrand">CARD</div>
        <small id="previewCvvLabel">CVV</small>
    </div>
    <div class="card-line fs-5" id="previewNumber">•••• •••• •••• ••••</div>
    <div class="d-flex justify-content-between mt-3">
        <div>
            <small class="d-block">Card Holder</small>
            <div id="previewName" class="fw-semibold">FULL NAME</div>
        </div>
        <div>
            <small class="d-block">Expires</small>
            <div id="previewExpiry" class="fw-semibold">MM/YY</div>
        </div>
    </div>
</div>

<script>
    (function() {
        // DOM refs
        const form = document.querySelector('form[action*="credit_card_payment"]');
        if (!form) return; // nothing to enhance

        const cardNumberInput = form.querySelector('#card_number');
        const cvvInput = form.querySelector('#cvv');
        const expiryInput = form.querySelector('#card_expiry_date');

        // helpers to show errors inline (Bootstrap invalid-feedback exists in your form markup)
        function setInvalid(input, message) {
            input.classList.add('is-invalid');
            // find or create invalid-feedback element
            let fb = input.parentElement.querySelector('.field-error, .invalid-feedback');
            if (!fb) {
                fb = document.createElement('div');
                fb.className = 'field-error';
                input.parentElement.appendChild(fb);
            }
            fb.textContent = message;
        }

        function clearInvalid(input) {
            input.classList.remove('is-invalid');
            let fb = input.parentElement.querySelector('.field-error');
            if (fb) fb.remove();
        }

        // Format card number: groups of 4 (but Amex groups 4-6-5). We'll do general space every 4 for simplicity
        function formatCardNumber(value) {
            return value.replace(/\D/g, '').substr(0, 19) // accept up to 19 digits (some cards)
                .replace(/(.{4})/g, '$1 ').trim();
        }

        // Luhn check for card number (digits only)
        function luhnCheck(numStr) {
            const arr = (numStr.split('').reverse()).map(x => parseInt(x, 10));
            let sum = 0;
            for (let i = 0; i < arr.length; i++) {
                let v = arr[i];
                if (i % 2 === 1) {
                    v *= 2;
                    if (v > 9) v -= 9;
                }
                sum += v;
            }
            return (sum % 10) === 0;
        }

        // Detect card brand (basic)
        function detectBrand(number) {
            const n = number.replace(/\D/g, '');
            if (/^4/.test(n)) return 'Visa';
            if (/^(5[1-5]|2[2-7])/.test(n)) return 'Mastercard';
            if (/^3[47]/.test(n)) return 'Amex';
            if (/^6(?:011|5|4[4-9])/.test(n)) return 'Discover';
            return 'Card';
        }

        // Format expiry to MM/YY
        function formatExpiry(value) {
            const digits = value.replace(/\D/g, '').slice(0, 4);
            if (digits.length <= 2) return digits;
            return digits.slice(0, 2) + '/' + digits.slice(2);
        }

        // Restrict inputs
        function onlyDigitsHandler(e) {
            // allow control keys
            if (['Backspace', 'ArrowLeft', 'ArrowRight', 'Delete', 'Tab'].includes(e.key)) return;
            if (!/\d/.test(e.key)) e.preventDefault();
        }

        // wire formatting events
        cardNumberInput.addEventListener('input', (e) => {
            const formatted = formatCardNumber(e.target.value);
            e.target.value = formatted;
            clearInvalid(cardNumberInput);
            updatePreview();
            updateBrandBehavior();
        });
        cardNumberInput.addEventListener('keydown', onlyDigitsHandler);

        expiryInput.addEventListener('input', (e) => {
            const old = e.target.value;
            e.target.value = formatExpiry(old);
            clearInvalid(expiryInput);
            updatePreview();
        });
        expiryInput.addEventListener('keydown', onlyDigitsHandler);

        cvvInput.addEventListener('input', (e) => {
            // limit length depending on brand
            const brand = detectBrand(cardNumberInput.value || '');
            const max = (brand === 'Amex') ? 4 : 3;
            e.target.value = e.target.value.replace(/\D/g, '').slice(0, max);
            clearInvalid(cvvInput);
        });
        cvvInput.addEventListener('keydown', onlyDigitsHandler);

        // Optionally show a visual preview
        const preview = document.getElementById('cardPreview');
        const previewNumber = document.getElementById('previewNumber');
        const previewExpiry = document.getElementById('previewExpiry');
        const previewBrand = document.getElementById('cardBrand');
        const previewName = document.getElementById('previewName');

        function updatePreview() {
            // show preview once user types something
            const num = cardNumberInput.value.trim();
            if (num.length) preview.classList.remove('d-none');
            else preview.classList.add('d-none');

            previewNumber.textContent = num ? num : '•••• •••• •••• ••••';
            previewExpiry.textContent = expiryInput.value || 'MM/YY';
            previewBrand.textContent = detectBrand(num);
            // optional: set name from auth user if you want; leaving placeholder
        }

        function updateBrandBehavior() {
            const brand = detectBrand(cardNumberInput.value || '');
            // adjust placeholder/CVV max length visually
            if (brand === 'Amex') {
                cvvInput.setAttribute('maxlength', 4);
                cvvInput.setAttribute('placeholder', '4 digits');
            } else {
                cvvInput.setAttribute('maxlength', 3);
                cvvInput.setAttribute('placeholder', '3 digits');
            }
            previewBrand.textContent = brand;
        }

        // Validate expiry not in past and proper MM/YY
        function validateExpiry(value) {
            if (!/^\d{2}\/\d{2}$/.test(value)) return false;
            const [mm, yy] = value.split('/').map(x => parseInt(x, 10));
            if (mm < 1 || mm > 12) return false;
            const now = new Date();
            const currentYear = now.getFullYear() % 100; // two-digit
            const currentMonth = now.getMonth() + 1;
            if (yy < currentYear) return false;
            if (yy === currentYear && mm < currentMonth) return false;
            return true;
        }

        // On submit: perform client-side checks (Luhn, expiry, cvv length)
        form.addEventListener('submit', function(e) {
            // clear previous errors
            [cardNumberInput, expiryInput, cvvInput].forEach(clearInvalid);

            const rawNumber = cardNumberInput.value.replace(/\s+/g, '');
            const brand = detectBrand(cardNumberInput.value || '');
            const cvvLen = (brand === 'Amex') ? 4 : 3;

            let hasError = false;

            // number presence & length
            if (!rawNumber || rawNumber.length < 12) {
                setInvalid(cardNumberInput, 'Please enter a valid card number.');
                hasError = true;
            } else if (!luhnCheck(rawNumber)) {
                setInvalid(cardNumberInput, 'Card number appears invalid (failed checksum).');
                hasError = true;
            }

            // expiry
            if (!validateExpiry(expiryInput.value || '')) {
                setInvalid(expiryInput, 'Please enter a valid expiry date (MM/YY) that is not expired.');
                hasError = true;
            }

            // cvv
            if (!cvvInput.value || cvvInput.value.length !== cvvLen) {
                setInvalid(cvvInput, `CVV must be ${cvvLen} digits.`);
                hasError = true;
            }

            if (hasError) {
                e.preventDefault();
                e.stopPropagation();
                // focus first invalid
                const firstInvalid = form.querySelector('.is-invalid');
                if (firstInvalid) firstInvalid.focus();
            } else {
                // allowed to submit — let Laravel handle the rest
                // (we don't modify field values; original inputs are sent)
            }
        });

        // initialize preview & brand behavior if values already present (old values)
        updatePreview();
        updateBrandBehavior();

    })();
</script>
<?php /**PATH C:\xampp\htdocs\projects\online-banking-v7.0\resources\views/dashboard/user/partials/credit_card_style_and_script.blade.php ENDPATH**/ ?>